<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Imagick\Driver;
use Intervention\Gif\Exceptions\NotReadableException;
use Illuminate\Support\Facades\File;

class ImageService
{
    public function createCollage($imagePaths,$title, $text = 'The Best Handheld Vacuums of 2025')
    {
        // 1. Initialize the Image Manager
        $manager = new ImageManager(new Driver());
        // $imagePaths = [
        //     public_path('71nUwIp+f2L.jpg'), 
        //     public_path('41X7CNEDPZL.jpg'),
        //     public_path('41lrx5UKGBL.jpg'),
        // ];
        // Font path check remains critical for the text overlay
        $fontPath = public_path('fonts/arial.ttf');

        // --- DYNAMIC DIMENSIONS ---
        $imageCount = count($imagePaths);
        if ($imageCount === 0) {
            return response()->json(['error' => 'No image paths provided for the collage.'], 400);
        }

        // Define base dimensions. We will adjust the width based on the number of images.
        // Let's ensure enough space for up to 4 images comfortably.
        $canvasHeight = 400;
        $padding = 10;

        // Calculate the required canvas width: (Image Width * N) + (Padding * (N + 1))
        // Assuming a max width per image of 240px for a total of 960px + padding.
        $maxImageWidth = 240;
        $canvasWidth = ($maxImageWidth * $imageCount) + ($padding * ($imageCount + 1));

        // Final calculated dimensions for each image slot
        $totalPaddingWidth = $padding * ($imageCount + 1);
        $availableWidth = $canvasWidth - $totalPaddingWidth;
        $imageWidth = floor($availableWidth / $imageCount); // floor to ensure integer size
        $imageHeight = $canvasHeight - ($padding * 2);

        $outputPath = public_path('collages/'.$title.'.png');
        // --- 2. Create the Base Canvas ---
        $collage = $manager->create($canvasWidth, $canvasHeight)->fill('#f0f0f0');

        // Check for the font file (as before)
        if (!file_exists($fontPath)) {
            Log::warning("Font file not found: $fontPath. Text overlay may fail.");
        }

        // --- 2. Create the Base Canvas ---
        try {
            // --- 3. LOOP through all images and place them ---
            foreach ($imagePaths as $index => $path) {

                // Calculate the X-offset for the current image
                // Offset starts at the first padding + (index * (imageWidth + padding))
                $offsetX = $padding + ($index * ($imageWidth + $padding));

                try {
                    // Read the image (supports URL or local path)
                    $img = $manager->read($path)
                        // Resize the image to fit the calculated slot size
                        ->resize($imageWidth, $imageHeight);

                    // Place the image on the canvas
                    $collage->place($img, 'top-left', $offsetX, $padding);
                } catch (NotReadableException $e) {
                    // Handle failure for a single image gracefully
                    Log::warning("Skipping image at index $index. Could not read path: $path. Details: " . $e->getMessage());

                    // Optional: Place a placeholder for the missing image
                    $placeholder = $manager->create($imageWidth, $imageHeight)->fill('#ff0000');
                    $collage->place($placeholder, 'top-left', $offsetX, $padding);
                }
            }

            // Optional: Add text (only if font exists)
            if (file_exists($fontPath)) {
                $collage->text($text, floor($canvasWidth / 2), $canvasHeight - 30, function ($font) use ($fontPath) {
                    $font->file($fontPath);
                    $font->size(28); // Adjusted size for potentially wider canvas
                    $font->color('#fff');
                    $font->stroke('#ff5500', 2);
                    $font->align('center');
                    $font->valign('bottom');
                });
            } else {
                Log::warning("Skipping text overlay because font file was not found.");
            }

            // --- 4. Save the result ---
            $collage->save($outputPath);

            return response()->json([
                'message' => 'Dynamic collage successfully created.',
                'path' => $outputPath,
                'image_count' => $imageCount,
                'canvas_size' => "{$canvasWidth}x{$canvasHeight}",
            ]);
        } catch (NotReadableException $e) {
            // Catch overall failure (e.g., if the font fails or canvas creation has an issue)
            return response()->json([
                'error' => 'A critical file (like the font) could not be read, or a file permission error occurred.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function downloadImage($url)
    {
        $filename = basename($url);
        $path = public_path('images/' . $filename);
        $image = file_get_contents($url);
        File::put($path, $image);
        return $filename;
    }
}
