<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;

class AmazonProductService
{
    protected $accessKey;
    protected $secretKey;
    protected $partnerTag;
    protected $marketplace;
    protected $region = 'us-east-1';
    protected $service = 'ProductAdvertisingAPI';
    protected $endpoint = 'webservices.amazon.com';

    public function __construct()
    {
        $this->marketplace = config('services.amazon.marketplace');

        $this->accessKey = config('services.amazon_pa.access_key');
        $this->secretKey = config('services.amazon_pa.secret_key');
        $this->partnerTag = config('services.amazon_pa.partner_tag');
        $this->endpoint = config('services.amazon.endpoint', 'webservices.amazon.com');
    }

    public function search($keywords, $limit = 10)
    {
        $payload = [
            "Keywords" => $keywords,
            "SearchIndex" => "All",
            "PartnerTag" => $this->partnerTag,
            "PartnerType" => "Associates",
            "Marketplace" => "www.amazon.com",
            'ItemCount' => min(10, max(1, $limit)),
            "Resources" => [
                "Images.Primary.Large",
                "ItemInfo.Title",
                "Offers.Listings.Price"
            ]
        ];

        $uri = "/paapi5/searchitems";
        $host = $this->endpoint;
        $amzDate = gmdate('Ymd\THis\Z');
        $dateStamp = gmdate('Ymd');
        $canonicalUri = $uri;
        $canonicalQueryString = '';
        $canonicalHeaders = "content-encoding:amz-1.0\nhost:{$host}\nx-amz-date:{$amzDate}\n";
        $signedHeaders = 'content-encoding;host;x-amz-date';
        $payloadJson = json_encode($payload);
        $hashedPayload = hash('sha256', $payloadJson);
        $canonicalRequest = "POST\n{$canonicalUri}\n{$canonicalQueryString}\n{$canonicalHeaders}\n{$signedHeaders}\n{$hashedPayload}";

        $algorithm = 'AWS4-HMAC-SHA256';
        $credentialScope = "{$dateStamp}/{$this->region}/{$this->service}/aws4_request";
        $stringToSign = "{$algorithm}\n{$amzDate}\n{$credentialScope}\n" . hash('sha256', $canonicalRequest);
        $signature = $this->getSignature($dateStamp, $stringToSign);

        $authorizationHeader = "{$algorithm} Credential={$this->accessKey}/{$credentialScope}, SignedHeaders={$signedHeaders}, Signature={$signature}";

        $response = Http::withHeaders([
            'Content-Encoding' => 'amz-1.0',
            'Content-Type' => 'application/json; charset=UTF-8',
            'Host' => $host,
            'X-Amz-Date' => $amzDate,
            'Authorization' => $authorizationHeader,
        ])->post("https://{$host}{$uri}", $payload);

        return $response->json();
    }

    private function getSignature($dateStamp, $stringToSign)
    {
        $kDate = hash_hmac('sha256', $dateStamp, 'AWS4' . $this->secretKey, true);
        $kRegion = hash_hmac('sha256', $this->region, $kDate, true);
        $kService = hash_hmac('sha256', $this->service, $kRegion, true);
        $kSigning = hash_hmac('sha256', 'aws4_request', $kService, true);
        return hash_hmac('sha256', $stringToSign, $kSigning);
    }

    function fetch_product_by_asin(string $asin)
    {
        $response = Http::withOptions(['allow_redirects' => true])
            ->timeout(260)             // How long to wait for full response
            ->connectTimeout(110)      // How long to wait for connection
            ->retry(3, 2000)
            ->get('https://api.rainforestapi.com/request', [
                'api_key' => config('services.rainforest.api_key'),
                'type' => 'search',
                'amazon_domain' => 'amazon.com',
                'asin' => $asin,
                'type' => 'product',
                'include_html' => 'false',
                'output' => 'json'
            ]);
        $data = $response->json();
        return $data;
    }
    // function fetch_products($keyword,$product_count=5){
    //     $response = Http::withOptions(['allow_redirects' => true])
    //         ->get('https://api.rainforestapi.com/request', [
    //             'api_key' => config('services.rainforest.api_key'),
    //             'type' => 'search',
    //             'amazon_domain' => 'amazon.com',
    //             'search_term' => $keyword,
    //             'include_products_count' => $product_count,
    //             'output' => 'json',
    //             'include_html' => 'false'
    //         ]);

    //     return $response->json();
    // }
    function fetch_products($keyword, $product_count = 5)
    {
        $response = Http::withOptions([
            'allow_redirects' => true,
        ])
            ->timeout(260)             // How long to wait for full response
            ->connectTimeout(110)      // How long to wait for connection
            ->retry(3, 2000)          // retry 3 times, wait 2s between
            ->get('https://api.rainforestapi.com/request', [
                'api_key' => config('services.rainforest.api_key'),
                'type' => 'search',
                'amazon_domain' => 'amazon.com',
                'search_term' => $keyword,
                'include_products_count' => $product_count,
                'output' => 'json',
                'include_html' => 'false'
            ]);

        return $response->json();
    }
}
