<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

/**
 * Minimal Amazon Product Advertising API v5 client for SearchItems.
 * Requires these env vars:
 * - AMAZON_PA_ACCESS_KEY
 * - AMAZON_PA_SECRET_KEY
 * - AMAZON_PA_ASSOCIATE_TAG
 * - AMAZON_PA_HOST (e.g., webservices.amazon.com or webservices.amazon.co.uk)
 * - AMAZON_PA_REGION (e.g., us-east-1)
 */
class AmazonProductAdvertisingService
{
    protected string $accessKey;
    protected string $secretKey;
    protected string $associateTag;
    protected string $host;
    protected string $region;

    public function __construct()
    {
        $this->accessKey = config('services.amazon_pa.access_key') ?: env('AMAZON_PA_ACCESS_KEY');
        $this->secretKey = config('services.amazon_pa.secret_key') ?: env('AMAZON_PA_SECRET_KEY');
        $this->associateTag = config('services.amazon_pa.associate_tag') ?: env('AMAZON_PA_ASSOCIATE_TAG');
        $this->host = config('services.amazon_pa.host') ?: env('AMAZON_PA_HOST', 'webservices.amazon.com');
        $this->region = config('services.amazon_pa.region') ?: env('AMAZON_PA_REGION', 'us-east-1');
    }

    /**
     * Search items by keywords. Returns array or throws exception on failure.
     * @param string $keywords
     * @param int $limit
     * @return array
     */
    public function searchItems(string $keywords, int $limit = 10): array
    {
        if (empty($this->accessKey) || empty($this->secretKey) || empty($this->associateTag)) {
            throw new \RuntimeException('Amazon PA credentials not configured (AMAZON_PA_ACCESS_KEY / AMAZON_PA_SECRET_KEY / AMAZON_PA_ASSOCIATE_TAG).');
        }

        $endpoint = "https://{$this->host}/paapi5/searchitems";

        $payload = [
            'Keywords' => $keywords,
            'SearchIndex' => 'All',
            'Resources' => [
                'ItemInfo.Title',
                'ItemInfo.ByLineInfo',
                'ItemInfo.ContentInfo',
                'ItemInfo.ProductInfo',
                'Offers.Listings.Price',
                'Images.Primary.Medium'
            ],
            'PartnerTag' => $this->associateTag,
            'PartnerType' => 'Associates',
            'Operation' => 'SearchItems',
            'ItemCount' => min(10, max(1, $limit)),
            'Availability' => 'Available'
        ];

        $payload = json_encode($payload, JSON_UNESCAPED_SLASHES);
        $headers = $this->signedHeaders('POST', '/paapi5/searchitems', $payload);

        $response = Http::withHeaders($headers)
            ->post($endpoint, json_decode($payload, true));

        // Log the complete request for debugging
        Log::debug('PA-API Request:', [
            'endpoint' => $endpoint,
            'headers' => $headers,
            'payload' => json_decode($payload, true),
            'response' => $response->json()
        ]);

        if ($response->failed()) {
            throw new \RuntimeException('Amazon PA API error: ' . $response->body());
        }

        return $response->json();
    }

    /**
     * Create AWS4 signed headers for PA-API v5 request.
     * This is a minimal signer and may not cover every edge case.
     * @param string $method
     * @param string $uri
     * @param string $payload
     * @return array
     */
    protected function signedHeaders(string $method, string $uri, string $payload): array
    {
        // Initialize request parameters
        $service = 'ProductAdvertisingAPI';
        $host = $this->host;
        $region = $this->region;
        $accessKey = $this->accessKey;
        $secretKey = $this->secretKey;

        // Get current time in UTC
        $dateTime = new \DateTime('now', new \DateTimeZone('UTC'));
        $amzDate = $dateTime->format('Ymd\THis\Z');
        $dateStamp = $dateTime->format('Ymd');
        
        // Required request headers in canonical order
        $canonicalHeaders = [
            'content-encoding' => 'amz-1.0',
            'content-type' => 'application/json; charset=utf-8',
            'host' => $host,
            'x-amz-date' => $amzDate,
            'x-amz-target' => 'com.amazon.paapi5.v1.ProductAdvertisingAPIv1.SearchItems'
        ];
        ksort($canonicalHeaders);

        // Create canonical request
        $payloadHash = hash('sha256', $payload);
        $canonicalUri = $uri;
        $canonicalQueryString = '';
        
        // Build canonical headers string
        $canonicalHeadersString = '';
        foreach ($canonicalHeaders as $key => $value) {
            $canonicalHeadersString .= strtolower($key) . ':' . trim($value) . "\n";
        }
        
        $signedHeadersString = implode(';', array_keys($canonicalHeaders));
        
        // Create canonical request
        $canonicalRequest = implode("\n", [
            strtoupper($method),
            $canonicalUri,
            $canonicalQueryString,
            $canonicalHeadersString,
            $signedHeadersString,
            $payloadHash
        ]);

        // Create signing data
        $algorithm = 'AWS4-HMAC-SHA256';
        $credentialScope = implode('/', [
            $dateStamp,
            $region,
            $service,
            'aws4_request'
        ]);

        // Create string to sign
        $stringToSign = implode("\n", [
            $algorithm,
            $amzDate,
            $credentialScope,
            hash('sha256', $canonicalRequest)
        ]);

        // Calculate signature
        $dateKey = hash_hmac('sha256', $dateStamp, "AWS4{$secretKey}", true);
        $regionKey = hash_hmac('sha256', $region, $dateKey, true);
        $serviceKey = hash_hmac('sha256', $service, $regionKey, true);
        $signingKey = hash_hmac('sha256', 'aws4_request', $serviceKey, true);
        $signature = hash_hmac('sha256', $stringToSign, $signingKey);

        // Create authorization header with exact spacing
        $authorization = $algorithm . 
            ' Credential=' . $accessKey . '/' . $credentialScope .
            ', SignedHeaders=' . $signedHeadersString .
            ', Signature=' . $signature;

        Log::debug('AWS Signature Debug', [
            'canonicalRequest' => $canonicalRequest,
            'stringToSign' => $stringToSign,
            'credentialScope' => $credentialScope,
            'signature' => $signature
        ]);

        return [
            'Authorization' => $authorization,
            'Content-Type' => 'application/json; charset=utf-8',
            'Content-Encoding' => 'amz-1.0',
            'Host' => $host,
            'X-Amz-Date' => $amzDate,
            'X-Amz-Target' => 'com.amazon.paapi5.v1.ProductAdvertisingAPIv1.SearchItems',
            'User-Agent' => 'Laravel/PA-API-5.0 (Language=PHP/8.1)'
        ];
    }
}
